function f_audio = sonify_chroma(f_chroma,stepsize_ms,fs,parameter)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Name: sonify_chroma.m 
% Version: 1
% Date: 28.08.2007
% Programmer: Sebastian Ewert
%
% Description:
% - uses sine tones to sonify a chroma series
% - for every pitch class multiple sines are played in parallel. One for
%   every octave. We start at C(-1) up to the nyquist frequency
%
% Input: 
%        f_chroma
%        stepsize_ms
%        fs
%        steptime_ms
%
%       parameter.use_all_octaves = 1
% 
% Output:
%        f_audio
%
% Example:
%  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if nargin<3
    error('You didn''t specify all required parameters');
end

if nargin<4
    parameter = [];
end

if isfield(parameter,'use_all_octaves')==0
    parameter.use_all_octaves = 1;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Main program
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% f_chroma : Each row is a chroma vector
lengthChroma = size(f_chroma,1);
lengthChroma_ms = round(lengthChroma * stepsize_ms);
samples_per_feature = round((stepsize_ms /1000) * fs);

f_audio = zeros( 1, lengthChroma * samples_per_feature);

maximum_midi_pitch_regarding_fs = floor(log2( fs/2 /440 ) *12 +69);


for pitch=1:12
    % we create f_audio_pitch a little bit too long and then cut it to the
    % exact length of f_audio
    f_audio_pitch = zeros(1,size((0:1/fs:(lengthChroma_ms+500)/1000),2));
    k=0;
    if parameter.use_all_octaves
        while ((pitch-1) + k*12) <= maximum_midi_pitch_regarding_fs
            pitch_freq_Hz = 2^( ( ((pitch-1) + k*12) - 69)/12)*440;
            f_audio_pitch = f_audio_pitch + sin( pitch_freq_Hz * 2*pi * (0:1/fs:(lengthChroma_ms+500)/1000) );
            k=k+1;
        end
    else
        % we use Octave 3 as center frequencies. C3 is midi pitch 48
        pitch_freq_Hz = 2^( ( ( 47 + pitch) - 69)/12)*440;
        f_audio_pitch = f_audio_pitch + sin( pitch_freq_Hz * 2*pi * (0:1/fs:(lengthChroma_ms+500)/1000) );
    end
    f_audio_pitch = f_audio_pitch(1:size(f_audio,2));
    
    f_audio_pitch_scaler = zeros(size(f_audio_pitch));
    for k=1:lengthChroma
        f_audio_pitch_scaler( (k-1)*samples_per_feature+1:k*samples_per_feature) = f_chroma(k,pitch);
    end
    
    % we smooth the hard transitions between feature vectors within 3ms
    smoothing_filter = window( @hamming, round(3/1000 * fs));
    smoothing_filter = smoothing_filter / sum(smoothing_filter);
    f_audio_pitch_scaler = conv(f_audio_pitch_scaler,smoothing_filter);
    f_audio_pitch_scaler = f_audio_pitch_scaler(floor(length(smoothing_filter)/2):end-ceil(length(smoothing_filter)/2));
    
    f_audio = f_audio + f_audio_pitch .* f_audio_pitch_scaler;
end

f_audio = f_audio / (max(abs(f_audio)) * 1.0001) ;

return;



